from pydantic import BaseModel, condecimal, validator
from typing import Optional, List
from datetime import datetime, timezone
from app.models.order import OrderStatusEnum
from zoneinfo import ZoneInfo


BELGRADE_TZ = ZoneInfo("Europe/Belgrade")


class OrderItemBase(BaseModel):
    product_id: int
    variant_id: int
    quantity: int
    customization_text: Optional[str] = None


class OrderItemCreate(OrderItemBase):
    pass


class OrderItemRead(OrderItemBase):
    id: int
    order_id: int
    unit_price: condecimal(max_digits=10, decimal_places=2)
    product_name: str
    variant_size: str
    total_price: condecimal(max_digits=10, decimal_places=2)
    admin_unit_price: condecimal(max_digits=10, decimal_places=2)
    admin_total_price: condecimal(max_digits=10, decimal_places=2)
    image_url: Optional[str] = None

    class Config:
        from_attributes = True


class OrderBase(BaseModel):
    child_id: int
    notes: Optional[str] = None


class OrderCreate(OrderBase):
    items: List[OrderItemCreate]


class OrderUpdate(BaseModel):
    status: Optional[OrderStatusEnum] = None
    notes: Optional[str] = None


class OrderRejection(BaseModel):
    reason: str


class OrderRead(OrderBase):
    id: int
    parent_id: int
    club_id: int
    status: OrderStatusEnum
    created_at: datetime
    updated_at: datetime
    total_amount: condecimal(max_digits=10, decimal_places=2)
    admin_total_amount: condecimal(max_digits=10, decimal_places=2)
    child_first_name: Optional[str] = None
    child_last_name: Optional[str] = None
    items: List[OrderItemRead]

    class Config:
        from_attributes = True

    @validator('created_at', 'updated_at', pre=False)
    def _to_belgrade(cls, v: datetime) -> datetime:
        if v is None:
            return v
        if v.tzinfo is None:
            v = v.replace(tzinfo=timezone.utc)
        return v.astimezone(BELGRADE_TZ)


class OrderListRead(BaseModel):
    id: int
    parent_id: int
    child_id: int
    club_id: int
    status: OrderStatusEnum
    created_at: datetime
    total_amount: condecimal(max_digits=10, decimal_places=2)
    admin_total_amount: condecimal(max_digits=10, decimal_places=2)
    child_first_name: Optional[str] = None
    child_last_name: Optional[str] = None

    class Config:
        from_attributes = True

    @validator('created_at', pre=False)
    def _to_belgrade(cls, v: datetime) -> datetime:
        if v is None:
            return v
        if v.tzinfo is None:
            v = v.replace(tzinfo=timezone.utc)
        return v.astimezone(BELGRADE_TZ)
