from sqlalchemy import Column, Integer, String, ForeignKey, DateTime, Enum, Numeric, Boolean, Text
from sqlalchemy.orm import relationship
from datetime import datetime
import enum
from app.db.base import Base


class OrderStatusEnum(str, enum.Enum):
    PENDING = "PENDING"  # Submitted by parent
    APPROVED = "APPROVED"  # Approved by trainer
    REJECTED = "REJECTED"  # Rejected by trainer
    IN_PRODUCTION = "IN_PRODUCTION"  # Work order created
    COMPLETED = "COMPLETED"  # Delivered/fulfilled
    CANCELLED = "CANCELLED"


class Order(Base):
    __tablename__ = "orders"
    
    id = Column(Integer, primary_key=True, index=True)
    parent_id = Column(Integer, ForeignKey("users.id"), nullable=False)
    child_id = Column(Integer, ForeignKey("children.id"), nullable=False)
    club_id = Column(Integer, ForeignKey("clubs.id"), nullable=False)
    status = Column(Enum(OrderStatusEnum), nullable=False, default=OrderStatusEnum.PENDING)
    created_at = Column(DateTime, nullable=False, default=datetime.utcnow)
    updated_at = Column(DateTime, nullable=False, default=datetime.utcnow, onupdate=datetime.utcnow)
    total_amount = Column(Numeric(10, 2), nullable=False)
    notes = Column(String, nullable=True)
    
    parent = relationship("User", foreign_keys=[parent_id])
    child = relationship("Child")
    club = relationship("Club")
    items = relationship("OrderItem", back_populates="order", cascade="all, delete-orphan")

    @property
    def admin_total_amount(self):
        """Sum using admin prices for admin-facing totals."""
        total = 0
        for item in self.items:
            admin_price = item.admin_unit_price
            total += admin_price * item.quantity
        return total

    @property
    def child_first_name(self):
        return self.child.first_name if self.child else None

    @property
    def child_last_name(self):
        return self.child.last_name if self.child else None


class OrderItem(Base):
    __tablename__ = "order_items"
    
    id = Column(Integer, primary_key=True, index=True)
    order_id = Column(Integer, ForeignKey("orders.id"), nullable=False)
    product_id = Column(Integer, ForeignKey("products.id"), nullable=False)
    variant_id = Column(Integer, ForeignKey("product_variants.id"), nullable=False)
    quantity = Column(Integer, nullable=False, default=1)
    unit_price = Column(Numeric(10, 2), nullable=False)  # Price at time of order
    customization_text = Column(Text, nullable=True)
    
    order = relationship("Order", back_populates="items")
    product = relationship("Product")
    variant = relationship("ProductVariant")
    
    @property
    def product_name(self):
        return self.product.name if self.product else ""
    
    @property
    def variant_size(self):
        return self.variant.size if self.variant else ""
    
    @property
    def image_url(self):
        return self.product.image_url if self.product else None
    
    @property
    def total_price(self):
        return self.unit_price * self.quantity

    @property
    def admin_unit_price(self):
        # Prefer variant.admin_price; fall back to unit_price if missing
        if self.variant and self.variant.admin_price is not None:
            return self.variant.admin_price
        return self.unit_price

    @property
    def admin_total_price(self):
        return self.admin_unit_price * self.quantity
