from fastapi import FastAPI
from fastapi.staticfiles import StaticFiles
from app.api.v1 import router as v1_router
from fastapi.middleware.cors import CORSMiddleware
from pathlib import Path
from sqlalchemy import inspect, text
from app.db.session import engine
import os

app = FastAPI(title="Sports Ordering API")

# Production-ready CORS: allow frontend and external access
allowed_origins = [
    "http://localhost:5173",
    "http://localhost:3000",
    "http://127.0.0.1:5173",
    "http://127.0.0.1:3000",
]

# Add environment-based origins (e.g., deployed frontend URL)
frontend_url = os.getenv("FRONTEND_URL")
if frontend_url:
    allowed_origins.append(frontend_url)

# Allow from any origin for image serving (optional: restrict if needed)
# Uncomment the line below for truly open access to images
# allowed_origins = ["*"]

app.add_middleware(
    CORSMiddleware,
    allow_origins=allowed_origins,
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Mount static files for serving uploaded images with caching
uploads_dir = Path("uploads")
uploads_dir.mkdir(exist_ok=True)
app.mount("/uploads", StaticFiles(directory="uploads"), name="uploads")

app.include_router(v1_router.router, prefix="/api/v1")


def ensure_products_columns() -> None:
    inspector = inspect(engine)
    if "products" not in inspector.get_table_names():
        return

    existing_columns = {col["name"] for col in inspector.get_columns("products")}
    dialect = engine.dialect.name

    statements: list[str] = []
    if "currency" not in existing_columns:
        if dialect == "postgresql":
            statements.append("ALTER TABLE products ADD COLUMN currency VARCHAR NOT NULL DEFAULT 'RSD'")
        else:
            statements.append("ALTER TABLE products ADD COLUMN currency VARCHAR NOT NULL DEFAULT 'RSD'")

    if "has_customization" not in existing_columns:
        if dialect == "postgresql":
            statements.append("ALTER TABLE products ADD COLUMN has_customization BOOLEAN NOT NULL DEFAULT FALSE")
        else:
            statements.append("ALTER TABLE products ADD COLUMN has_customization BOOLEAN NOT NULL DEFAULT 0")

    if statements:
        with engine.begin() as connection:
            for statement in statements:
                connection.execute(text(statement))


@app.on_event("startup")
def startup_migrations() -> None:
    ensure_products_columns()

@app.get("/")
def root():
    return {"status": "ok"}
