from fastapi import APIRouter, Depends, HTTPException, status, Query, UploadFile, File, Form, Request
from sqlalchemy.orm import Session, joinedload
from typing import List, Optional
from app.core import deps
from app.models import Product, ProductVariant, User, Club, RoleEnum
from app.schemas import product as product_schemas
from app.utils.file_storage import save_product_image, delete_product_image
import json

router = APIRouter()


# ===== ADMIN ENDPOINTS =====

@router.post("/", response_model=product_schemas.ProductRead, status_code=status.HTTP_201_CREATED)
async def create_product(
    request: Request,
    club_id: int = Form(...),
    name: str = Form(...),
    description: Optional[str] = Form(None),
    custom_fields: Optional[str] = Form(None),  # JSON string of custom fields
    variants: str = Form(...),  # JSON string of variants: [{size: "S", price: 100}, ...]
    has_customization: Optional[str] = Form("false"),
    currency: Optional[str] = Form("RSD"),
    image: Optional[UploadFile] = File(None),
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Create a new product with variants (admin only)"""
    import logging
    logger = logging.getLogger(__name__)
    logger.info(f"Creating product - club_id: {club_id}, name: {name}, variants: {variants}")
    
    # Verify club exists
    club = db.query(Club).filter(Club.id == club_id).first()
    if not club:
        logger.error(f"Club {club_id} not found")
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
    
    # Admin can only create products for clubs they manage
    if club.admin_id != current_user.id:
        logger.error(f"Admin {current_user.id} does not manage club {club_id}")
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only create products for clubs you manage"
        )
    
    # Parse variants JSON - expecting [{size: "S", price: 100}, {size: "M", price: 150}, ...]
    try:
        variant_list = json.loads(variants)
        logger.info(f"Parsed variants: {variant_list}")
        if not isinstance(variant_list, list) or len(variant_list) == 0:
            logger.error("Variants is not a non-empty list")
            raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Variants must be a non-empty list")
    except json.JSONDecodeError as e:
        logger.error(f"JSON decode error: {e}")
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Invalid variants format")
    except Exception as e:
        logger.error(f"Unexpected error parsing variants: {e}")
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=f"Error parsing variants: {str(e)}")
    
    # Handle image upload
    image_path = None
    if image:
        try:
            image_path = await save_product_image(image)
            logger.info(f"Image saved to: {image_path}")
        except ValueError as e:
            logger.error(f"Image upload error: {e}")
            raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    
    has_customization_value = str(has_customization).lower() in {"true", "1", "yes", "on"}

    # Create product
    try:
        product = Product(
            club_id=club_id,
            name=name,
            description=description,
            image_url=image_path,
            is_confirmed=False,
            custom_fields=custom_fields,
            has_customization=has_customization_value
        )
        db.add(product)
        db.flush()  # Get product.id
        logger.info(f"Product created with id: {product.id}")
        
        # Create variants - each with their own price
        for variant_data in variant_list:
            variant = ProductVariant(
                product_id=product.id,
                size=variant_data.get('size', 'Unknown'),
                admin_price=variant_data.get('price', 100)
            )
            db.add(variant)
            logger.info(f"Variant created: {variant.size} - {variant.admin_price}")
        
        db.commit()
        db.refresh(product)
        logger.info("Product and variants committed successfully")
        return product
    except Exception as e:
        db.rollback()
        logger.error(f"Database error: {e}")
        raise HTTPException(status_code=status.HTTP_500_INTERNAL_SERVER_ERROR, detail=f"Database error: {str(e)}")


@router.put("/{product_id}", response_model=product_schemas.ProductRead)
def update_product(
    product_id: int,
    product_in: product_schemas.ProductUpdate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Update a product (admin only)"""
    product = db.query(Product).filter(Product.id == product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    # Verify admin manages this club
    if product.club.admin_id != current_user.id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only update products for clubs you manage"
        )
    
    update_data = product_in.model_dump(exclude_unset=True)
    for field, value in update_data.items():
        setattr(product, field, value)
    
    db.commit()
    db.refresh(product)
    return product


@router.delete("/{product_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_product(
    product_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Delete a product (admin only)"""
    product = db.query(Product).filter(Product.id == product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    # Verify admin manages this club
    if product.club.admin_id != current_user.id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only delete products for clubs you manage"
        )
    
    # Delete associated image file
    if product.image_url:
        delete_product_image(product.image_url)
    
    db.delete(product)
    db.commit()
    return None


@router.delete("/club/{club_id}/all", status_code=status.HTTP_204_NO_CONTENT)
def delete_all_club_products(
    club_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Delete all products for a specific club (admin only)"""
    club = db.query(Club).filter(Club.id == club_id).first()
    if not club:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
    
    # Verify admin manages this club
    if club.admin_id != current_user.id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only delete products for clubs you manage"
        )
    
    # Delete all products for this club
    db.query(Product).filter(Product.club_id == club_id).delete()
    db.commit()
    return None


@router.get("/admin/list", response_model=List[product_schemas.ProductRead])
def list_admin_products(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin),
    club_id: Optional[int] = Query(None),
    skip: int = 0,
    limit: int = 100
):
    """List all products for clubs managed by admin"""
    managed_club_ids = [club.id for club in current_user.managed_clubs]
    if not managed_club_ids:
        return []
    
    query = db.query(Product).options(
        joinedload(Product.variants)
    ).filter(Product.club_id.in_(managed_club_ids))
    
    if club_id:
        if club_id not in managed_club_ids:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail="You don't manage this club"
            )
        query = query.filter(Product.club_id == club_id)
    
    products = query.offset(skip).limit(limit).all()
    return products


@router.delete("/{product_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_product(
    product_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Delete a product (admin only)"""
    product = db.query(Product).filter(Product.id == product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    db.delete(product)
    db.commit()
    return None


# Variant endpoints
@router.post("/{product_id}/variants", response_model=product_schemas.ProductVariantRead, status_code=status.HTTP_201_CREATED)
def create_variant(
    product_id: int,
    variant_in: product_schemas.ProductVariantCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Add a variant to an existing product (admin only)"""
    product = db.query(Product).filter(Product.id == product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    variant = ProductVariant(**variant_in.model_dump(), product_id=product_id)
    db.add(variant)
    db.commit()
    db.refresh(variant)
    return variant


@router.put("/variants/{variant_id}", response_model=product_schemas.ProductVariantRead)
def update_variant(
    variant_id: int,
    variant_in: product_schemas.ProductVariantUpdate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Update a product variant (admin only)"""
    variant = db.query(ProductVariant).filter(ProductVariant.id == variant_id).first()
    if not variant:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Variant not found")
    
    update_data = variant_in.model_dump(exclude_unset=True)
    for field, value in update_data.items():
        setattr(variant, field, value)
    
    db.commit()
    db.refresh(variant)
    return variant


@router.delete("/variants/{variant_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_variant(
    variant_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Delete a product variant (admin only)"""
    variant = db.query(ProductVariant).filter(ProductVariant.id == variant_id).first()
    if not variant:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Variant not found")
    
    db.delete(variant)
    db.commit()
    return None


# ===== TRAINER ENDPOINTS =====

@router.get("/trainer/list", response_model=List[product_schemas.ProductRead])
def list_trainer_products(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer),
    skip: int = 0,
    limit: int = 100
):
    """List all products for trainer's club (trainer only)"""
    if not current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Trainer not assigned to a club"
        )
    
    products = db.query(Product).options(
        joinedload(Product.variants)
    ).filter(
        Product.club_id == current_user.club_id
    ).offset(skip).limit(limit).all()
    
    return products


@router.put("/trainer/{product_id}/confirm", response_model=product_schemas.ProductRead)
def confirm_product(
    product_id: int,
    confirm_data: product_schemas.ProductConfirm,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """Confirm product to make it visible to parents (trainer only)"""
    product = db.query(Product).filter(Product.id == product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    # Verify product belongs to trainer's club
    if product.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only confirm products for your club"
        )
    
    product.is_confirmed = confirm_data.is_confirmed
    db.commit()
    db.refresh(product)
    return product


@router.put("/trainer/variants/{variant_id}/price", response_model=product_schemas.ProductVariantRead)
def set_variant_trainer_price(
    variant_id: int,
    price_data: product_schemas.ProductVariantSetPrice,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """Set trainer price for a variant (can only be higher than admin price)"""
    variant = db.query(ProductVariant).filter(ProductVariant.id == variant_id).first()
    if not variant:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Variant not found")
    
    # Verify product belongs to trainer's club
    if variant.product.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only set prices for products in your club"
        )
    
    # Ensure trainer price is not lower than admin price
    if price_data.trainer_price < variant.admin_price:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"Trainer price must be at least {variant.admin_price}"
        )
    
    variant.trainer_price = price_data.trainer_price
    db.commit()
    db.refresh(variant)
    return variant


# ===== PARENT ENDPOINTS =====

@router.get("/parent/list", response_model=List[product_schemas.ProductRead])
def list_parent_products(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent),
    club_id: Optional[int] = Query(None),
    skip: int = 0,
    limit: int = 100
):
    """List all confirmed products for parent's clubs (parent only)"""
    # Parents can see confirmed products from clubs they're associated with
    query = db.query(Product).options(
        joinedload(Product.variants)
    ).filter(Product.is_confirmed == True)
    
    if club_id:
        query = query.filter(Product.club_id == club_id)
    elif current_user.club_id:
        query = query.filter(Product.club_id == current_user.club_id)
    
    products = query.offset(skip).limit(limit).all()
    return products


@router.get("/{product_id}", response_model=product_schemas.ProductRead)
def get_product(
    product_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_user)
):
    """Get a specific product by ID with variants"""
    product = db.query(Product).filter(Product.id == product_id).first()
    if not product:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    # Parents can only see confirmed products
    if current_user.role == RoleEnum.PARENT and not product.is_confirmed:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Product not found")
    
    # Trainers can only see products from their club
    if current_user.role == RoleEnum.TRAINER and product.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only view products for your club"
        )
    
    # Admins can only see products from clubs they manage
    if current_user.role == RoleEnum.COMPANY_EMPLOYEE:
        managed_club_ids = [club.id for club in current_user.managed_clubs]
        if product.club_id not in managed_club_ids:
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail="You can only view products for clubs you manage"
            )
    
    return product


