from fastapi import APIRouter, Depends, HTTPException, status
from sqlalchemy.orm import Session
from typing import List
from app.core import deps
from app.models import Child, User
from pydantic import BaseModel
from datetime import datetime

router = APIRouter()


class ChildRead(BaseModel):
    id: int
    first_name: str
    last_name: str
    birth_year: int | None
    parent_id: int
    parent_email: str
    parent_name: str
    club_id: int

    class Config:
        from_attributes = True


class ChildCreate(BaseModel):
    first_name: str
    last_name: str
    birth_year: int


@router.get("/", response_model=List[ChildRead])
def list_children(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """List all children in trainer's club"""
    if not current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Trainer not assigned to a club"
        )
    
    children = db.query(Child).filter(Child.club_id == current_user.club_id).all()
    
    result = []
    for child in children:
        parent = db.query(User).filter(User.id == child.parent_id).first()
        result.append({
            "id": child.id,
            "first_name": child.first_name,
            "last_name": child.last_name,
            "birth_year": child.birth_year,
            "parent_id": child.parent_id,
            "parent_email": parent.email if parent else "",
            "parent_name": f"{parent.first_name} {parent.last_name}" if parent and parent.first_name else "",
            "club_id": child.club_id
        })
    
    return result


@router.get("/{child_id}", response_model=ChildRead)
def get_child(
    child_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_trainer)
):
    """Get specific child details"""
    child = db.query(Child).filter(Child.id == child_id).first()
    
    if not child:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Child not found")
    
    # Verify child is in trainer's club
    if child.club_id != current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="You can only view children in your club"
        )
    
    parent = db.query(User).filter(User.id == child.parent_id).first()
    
    return {
        "id": child.id,
        "first_name": child.first_name,
        "last_name": child.last_name,
        "birth_year": child.birth_year,
        "parent_id": child.parent_id,
        "parent_email": parent.email if parent else "",
        "parent_name": f"{parent.first_name} {parent.last_name}" if parent and parent.first_name else "",
        "club_id": child.club_id
    }


@router.get("/parent/my-children", response_model=List[ChildRead])
def list_my_children(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """List all children for the current parent"""
    children = db.query(Child).filter(Child.parent_id == current_user.id).all()
    
    result = []
    for child in children:
        result.append({
            "id": child.id,
            "first_name": child.first_name,
            "last_name": child.last_name,
            "birth_year": child.birth_year,
            "parent_id": child.parent_id,
            "parent_email": current_user.email,
            "parent_name": f"{current_user.first_name} {current_user.last_name}" if current_user.first_name else "",
            "club_id": child.club_id
        })
    
    return result


@router.post("/parent/add-child", response_model=ChildRead)
def add_child(
    payload: ChildCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Add a new child for the current parent"""
    if not current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Parent not assigned to a club"
        )
    
    child = Child(
        first_name=payload.first_name,
        last_name=payload.last_name,
        birth_year=payload.birth_year,
        parent_id=current_user.id,
        club_id=current_user.club_id
    )
    db.add(child)
    db.commit()
    db.refresh(child)
    
    return {
        "id": child.id,
        "first_name": child.first_name,
        "last_name": child.last_name,
        "birth_year": child.birth_year,
        "parent_id": child.parent_id,
        "parent_email": current_user.email,
        "parent_name": f"{current_user.first_name} {current_user.last_name}" if current_user.first_name else "",
        "club_id": child.club_id
    }


@router.delete("/parent/children/{child_id}")
def delete_child(
    child_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_parent)
):
    """Delete a child (only parent can delete their own children)"""
    child = db.query(Child).filter(
        Child.id == child_id,
        Child.parent_id == current_user.id
    ).first()
    
    if not child:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Child not found or not owned by you"
        )
    
    db.delete(child)
    db.commit()
    
    return {"message": "Child deleted successfully"}
